(ns vigilia-objects.core
  (:require [clojure.string :as s]))

(defn bacnet-is-binary? 
  "Return true (the object-type) if the object has binary values."
  [object-type]
  (some #{(str object-type)} ["3" "4" "5"]))



(defprotocol IVigiliaObject
  (id [this] "Return a globally unique ID for this object.")
  (short-id [this] "Return a per-project unique ID")
  (binary? [this] "Return true if the object is binary.")
  (value 
    [this]
    [this id-value-map]
    "Return the current value associated with the object (if any)."))



(defn print-vigilia-object [object]
  (str "#<Vigilia: " (into {} object)">"))


(defrecord VigiliaObject [project-id device-id object-type object-instance]
  IVigiliaObject
  (id [this] (s/join "." [project-id device-id object-type object-instance]))
  (short-id [this] (s/join "." [device-id object-type object-instance]))
  (binary? [this] (bacnet-is-binary? object-type))
  (value [this] (:present-value this 0)))


#?(:clj (defmethod clojure.core/print-method VigiliaObject [o writer]
          (.write writer (print-vigilia-object o))))



(defn vigilia-object
  "Generate a Vigilia object from a map or an ID. If a short ID is
  given, it must also be provided with the project-id as a second
  argument."
  ([map-or-id]
   (if (string? map-or-id)
     (let [splitted (s/split map-or-id #"\.")]
       (assert (= (count splitted) 4) "Missing element in the ID.")
       (let [[p-id d-id obj-type obj-inst] (s/split map-or-id #"\.")]
         (VigiliaObject. p-id d-id obj-type obj-inst)))
     (map->VigiliaObject map-or-id)))
  ([short-id project-id]
   (vigilia-object (str project-id "." short-id))))



;;;; virtual objects

(declare construct-operation-fn)

(defprotocol IVigiliaVirtualObject
  (source-objects [this] "Recursively find the source Vigilia objects. Return a collection of ID.")
  (operation [this] "Return the virtual object operation."))


        

(defn print-vigilia-virtual-object [object]
  (str "#<Vigilia virtual: " (into {} object)">"))



(defrecord VigiliaVirtualObject [object-name binary operation]
  IVigiliaObject
  (id [this] object-name)
  (short-id [this] (id this))
  (binary? [this] binary)
  (value [this id-value-map]
    ((construct-operation-fn this) id-value-map))

                                
  IVigiliaVirtualObject
  (operation [this] operation)
  (source-objects [this]
    (let [so-fn 
          (fn so-fn [op]
            (->> (drop 1 op) ;; drop the operator
                 (remove number?)
                 (mapcat (fn [o]
                           (cond
                             (satisfies? IVigiliaVirtualObject o) (source-objects o)
                             (satisfies? IVigiliaObject o) [o]
                             (coll? o) (so-fn o)
                             :else [o])))
                 (into #{})))]
      (so-fn operation))))





#?(:clj (defmethod clojure.core/print-method VigiliaVirtualObject [o writer]
          (.write writer (print-vigilia-virtual-object o))))




(def allowed-operators {"+" +
                        "-" -
                        "*" *
                        "/" /})

(defn construct-operation-fn
  "Return a function expecting a map of object IDs and values."
  [vir-o]
  (let [op-fn (fn op-fn [o]
                (let [args-fn
                      (->> (for [arg (drop 1 o)]
                             (cond
                               (number? arg) (fn [_] arg) ;; return the constant

                               ;; if we have an ID, substitute it by its value
                               (string? arg) #(get % arg "Missing source object!")                             

                               (satisfies? IVigiliaVirtualObject arg) (construct-operation-fn arg)
                               
                               (satisfies? IVigiliaObject arg) #(get % (id arg) "Missing object!")

                               (coll? arg) (op-fn arg)))
                           (into []))] ;; not lazy
                  (fn [data]
                    (let [result (apply (get allowed-operators (first o)) ;; operator fn
                                        (map #(% data) args-fn))]
                      (if (binary? vir-o)
                        (if (> result 0) 1 0)
                        result)))))]
    (op-fn (operation vir-o))))




(defn virtual-object
  "Generate a new virtual object. Map fields are :object-name, :binary
  and :operation."
  [object-map]
  (map->VigiliaVirtualObject (merge {:object-name "Virtual"
                                     :virtual true
                                     :operation ["+"]}
                                    object-map)))


(defn virtual?
  "Return true if the object is virtual."
  [obj]
  (when (:virtual obj)
    true))
  


(defn reconstruct-object
  "Given some object info (ID or map), will try to dispatch on the
  right object type."
  [o]
  (if (:virtual o)
    (virtual-object o)
    (vigilia-object o)))
  


(comment
  ;; testing
  (do
    (def obj-map {"51929055e4b05622e2d6603d.10600.0.1" 12
                  "51929055e4b05622e2d6603d.10600.0.2" 2})


    (def obj1 (vigilia-object "51929055e4b05622e2d6603d.10600.0.1"))
    (def obj2 (vigilia-object "51929055e4b05622e2d6603d.10600.0.2"))
    (def vir1 (virtual-object {:object-name "vir1" :operation ["+" obj1 obj2]}))
    (def vir2 (virtual-object {:object-name "vir1" :operation ["+" vir1 obj2]}))

    ((construct-operation-fn vir1) obj-map))
)
