(ns vigilia-objects.bacnet
  (:require [clojure.string :as string]))

;;; some bacnet related functions.

(defn is-binary? 
  "Return true (the object-type) if the object has binary values."
  [object-type]
  (some #{(str object-type)} ["3" "4" "5"]))

(def int-object-map
  {"0" :analog-input,
   "7" :command,
   "20" :trend-log,
   "27" :trend-log-multiple,
   "1" :analog-output,
   "24" :pulse-converter,
   "4" :binary-output,
   "15" :notification-class,
   "21" :life-safety-point,
   "13" :multi-state-input,
   "22" :life-safety-zone,
   "29" :structured-view,
   "6" :calendar,
   "28" :load-control,
   "25" :event-log,
   "17" :schedule,
   "3" :binary-input,
   "12" :loop,
   "2" :analog-value,
   "23" :accumulator,
   "19" :multi-state-value,
   "11" :group,
   "9" :event-enrollment,
   "5" :binary-value,
   "14" :multi-state-output,
   "16" :program,
   "30" :access-door,
   "10" :file,
   "18" :averaging,
   "8" :device})





(defn int-to-object-name [int]
  (or (some-> (get int-object-map (str int))
              (name)
              (string/capitalize)
              (string/replace #"-" " "))
      (str "unknown-"int)))

(defn object-name-to-int [obj-name]
  (when obj-name
    (or (let [k (some-> obj-name
                        (string/lower-case)
                        (string/replace #" " "-")
                        (keyword))]
          (some #(if (= (val %) k) (key %)) int-object-map))
        (nth (re-find #"unknown-(\d+)" obj-name) 1))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Object reference encoding ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; The following object refs will be encoded...
(comment
  (def example-objects-smaps
    [{:project-id "50b390c2e4b0e9e75d33e25c"
      :device-id "10200"
      :object-type "0"
      :object-instance "1"}
     {:project-id "50b390c2e4b0e9e75d33e25c"
      :device-id "10200"
      :object-type "0"
      :object-instance "2"}
     {:project-id "50b390c2e4b0e9e75d33e25c"
      :device-id "10100"
      :object-type "12"
      :object-instance "2"}
     {:project-id "50b390c2e4b0e9e75d33e25c"
      :device-id "10100"
      :object-type "5"
      :object-instance "15"}
     {:project-id "50b390c2e4b0e9e75d33e251"
      :device-id "10100"
      :object-type "5"
      :object-instance "14"}])

  (encoded-objects-smaps example-objects-smaps))
;; ===>
;;      [["50b390c2e4b0e9e75d33e25c"
;;        ["10200" "0.1" "0.2"]
;;        ["10100" "12.2" "5.15"]]
;;       ["50b390c2e4b0e9e75d33e251" ["10100" "5.14"]]]


(defn where
  "Use with `filter'"
  [criteria]
    (fn [m]
      (every? (fn [[k v]] (= (k m) v)) criteria)))

(defn encode-objects-smaps [objects-info]
  (into []
        (for [project (distinct (map :project-id objects-info))]
          (into [project]
                (let [devices (filter (where {:project-id project}) objects-info)]
                  (for [device (distinct (map :device-id devices))]
                    (into [device]
                          (let [objects (filter (where {:device-id device}) devices)]
                            (for [object objects]
                              (str (:object-type object)"."(:object-instance object)))))))))))


(defn decode-objects-smaps [encoded-objects-smaps]
  (->> (apply concat
             (for [project encoded-objects-smaps]
               (apply concat
                      (for [device (drop 1 project)]
                        (for [object (drop 1 device)]
                          (let [[t i] (clojure.string/split object #"\.")]
                            {:project-id (first project)
                             :device-id (first device)
                             :object-type t
                             :object-instance i}))))))
      (into [])))

